// //////////////////////////////////////////////////////////
// MemMapFile.cpp

#include "stdafx.h"
#include "MemMapFile.h"


// OS-specific
// Windows
#include <windows.h>


/// open file
bool MemMapFile::open(const wchar_t* filename)
{
  // already open ?
  if (isValid())
   return false;

  DWORD winHint = FILE_FLAG_RANDOM_ACCESS; // FILE_ATTRIBUTE_NORMAL, FILE_ATTRIBUTE_SEQUENTIAL_SCAN

  // open file
  _file = ::CreateFile(filename, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, winHint, NULL);
  if (_file==INVALID_HANDLE_VALUE)
    return false;

  // file size
  LARGE_INTEGER result;
  if (!GetFileSizeEx(_file, &result))
    return false;
  _filesize = static_cast<uint64_t>(result.QuadPart);

  // convert to mapped mode
  _mappedFile = ::CreateFileMapping(_file, NULL, PAGE_READONLY, 0, 0, NULL);
  if (!_mappedFile)
    return false;

  // initial mapping
  remap(0, _mappedBytes);

  if (!_mappedView)
    return false;

  // everything's fine
  return true;
}


/// close file
void MemMapFile::close()
{
  // kill pointer
  if (_mappedView)
  {
    ::UnmapViewOfFile(_mappedView);
    _mappedView = NULL;
  }

  if (_mappedFile)
  {
    ::CloseHandle(_mappedFile);
    _mappedFile = NULL;
  }

  // close underlying file
  if (_file !=INVALID_HANDLE_VALUE)
  {
    ::CloseHandle(_file);
    _file = INVALID_HANDLE_VALUE;
  }

  _filesize = 0;
}

/// raw access
const unsigned char* MemMapFile::getData(size_t offset/*=0*/) const
{
  //TODO: range check?
  return (const unsigned char*)(_mappedView+offset);
}


/// replace mapping by a new one of the same file, offset MUST be a multiple of the page size
bool MemMapFile::remap(uint64_t offset, size_t mappedBytes)
{
  if (!_filesize)
    return false;

  if (mappedBytes == 0)
    mappedBytes = _filesize;

  // close old mapping
  if (_mappedView)
  {
    ::UnmapViewOfFile(_mappedView);
    _mappedView = NULL;
  }

  // don't go further than end of file
  if (offset > _filesize)
    return false;
  if (offset + mappedBytes > _filesize)
    mappedBytes = size_t(_filesize - offset);

  DWORD offsetLow  = DWORD(offset & 0xFFFFFFFF);
  DWORD offsetHigh = DWORD(offset >> 32);
  _mappedBytes = mappedBytes;

  // get memory address
  _mappedView = (unsigned char *)::MapViewOfFile(_mappedFile, FILE_MAP_READ, offsetHigh, offsetLow, mappedBytes);

  if (_mappedView == NULL)
  {
    _mappedBytes = 0;
    _mappedView  = NULL;
    return false;
  }

  return true;
}


/// get OS page size (for remap)
int MemMapFile::getpagesize()
{
  SYSTEM_INFO sysInfo;
  GetSystemInfo(&sysInfo);
  return sysInfo.dwAllocationGranularity;
}
